# The functions below were copied from AnnotationDbi (where they are used
# internally and not exported), to be used as helpers in this package
dbEasyQuery <- function(conn, SQL, j0 = NA) {
    data0 <- DBI::dbGetQuery(conn, SQL)
    if (is.na(j0)) {
        return(data0)
    }
    ## Needed to deal properly with data frame with 0 column ("NULL data
    ## frames with 0 rows") returned by RSQLite when the result of a SELECT
    ## query has 0 row
    if (nrow(data0) == 0L) {
        character(0)
    } else {
        data0[[j0]]
    }
}

#' Add Target Full Name information to data frame
#' 
#' @description
#' Retrieve the Target Full Name for a given SeqId (or set of SeqIds) and
#' add that information to a data frame. The Target Full Name can be found
#' in an ADAT file and represents the full name of the protein target of a 
#' primary SOMAmer reagent.
#'
#' Note: You can retrieve a list of Target Full Name values for a given set of 
#' SeqIds with `mget(seqids, SomaScanTARGETFULLNAME)`, where `seqids` is a 
#' character vector of SeqIds. To retrieve a data frame of _all_ Target Full
#' Name values present in the package, use `toTable(SomaScanTARGETFULLNAME)`.
#' 
#' @param data a data frame generated by a `select` query. This data frame can 
#'    have any number of columns, but **must** contain a PROBEID column. This 
#'    will be used to match the Target Full Name. 
#' @returns The input data frame with one additional column 
#'    called "TARGETFULLNAME".
#' @examples
#' seqs <- withr::with_seed(123, sample(keys(SomaScan.db), 5))
#' res <- select(SomaScan.db, keys = seqs, columns = "UNIPROT")
#' addTargetFullName(res)
addTargetFullName <- function(data) {
    seqIds <- data$PROBEID
    targNames <- toTable(SomaScanTARGETFULLNAME)
    targNames <- targNames[targNames$probe_id %in% seqIds,]
    names(targNames) <- c("PROBEID", "TARGETFULLNAME")
    merge(targNames, data)
}

##############################################################

# The functions below were copied & modified (with permission) from
# SomaDataIO on CRAN, to avoid introducing new dependencies
`%@@%` <- function(x, y) {
    name <- as.character(substitute(y))
    attr(x, which = name, exact = TRUE)
}

locateSeqId <- function(x, trailing = TRUE) {
    pattern <- "[0-9]{4,5}[-.][0-9]{1,3}([._][0-9]{1,3})?$"
    if (!trailing) {
        pattern <- strtrim(pattern, nchar(pattern) - 1) # trim `"$"`
    }
    regex <- regexpr(pattern, x)
    start <- as.integer(regex)
    start[start < 0] <- NA_integer_
    stop <- start + (regex %@@% "match.length") - 1L
    data.frame(x = x, start = start, stop = stop, stringsAsFactors = FALSE)
}

getSeqId <- function(x) {
    # factor --> character; list --> character
    # zap trailing/leading whitespace
    df <- locateSeqId(trimws(x))
    seqId <- substr(df$x, df$start, df$stop)
    seqId <- sub("\\.", "-", seqId) # 1st '.' -> '-'
    seqId <- sub("\\.", "_", seqId) # 2nd '.' -> '_' if present
    seqId <- vapply(strsplit(seqId, "_", fixed = TRUE), `[[`, i = 1L, "")
    seqId
}
